// =============================================================
// Toplevel-Modul for FORTH-Processor K1
//
// Author   : Klaus Kohl-Schoepe
// Date     : 11.05.2020
// File Name: MAX1000_K1.v
// Copyright (C) 2020 Klaus Kohl-Schoepe (kks@designin.de)
// =============================================================

`timescale 1 ns / 1 ps
`default_nettype wire
`define MEMADDR 16      // 32K Bytes program memory + 32K Byte I/O
`define WIDTH 16        // 16 bit data size

// =============================================================
// Top-Modul MAX1000_K1
//  Interfaces: RX/TX (115200 Baud UART), 8 LED, 12MHz Clock
// =============================================================

module MAX1000_K1(
  input  CLK12M,        // 12MHz System Clock
  input  USER_BTN,      // User Button
  inout  [5:0] BDBUS,   // auch UART
  output [7:0] LED,     // LEDS
  inout  [7:0] PIO,     // PMOD Connector
  input  SEN_INT1,      // Interrupt from Accelerometer
  input  SEN_INT2,      // Interrupt from Accelerometer
  output SEN_SDI,       // Data to Accelerometer
  input  SEN_SDO,       // Data from Accelerometer
  output SEN_SPC,       // Clock to Accelerometer
  output SEN_CS         // Chip select to Accelerometer
);

// =============================================================
// A 32 Bit Counter with Processor speed
// =============================================================
  reg [31:0] counter = 32'b0;
 
// =============================================================
// LED port
// =============================================================
  reg  [7:0] ledreg;             // Register for LEDs
  assign LED = ledreg;
  
// =============================================================
// PMOD port
// =============================================================
  reg [7:0] pmod_data = 8'b0;     // PMOD data
  reg [7:0] pmod_dir = 8'b0;      // PMOD direction (0 = Input)
  assign PIO[0] = pmod_dir[0] ? pmod_data[0] : 1'bz;
  assign PIO[1] = pmod_dir[1] ? pmod_data[1] : 1'bz;
  assign PIO[2] = pmod_dir[2] ? pmod_data[2] : 1'bz;
  assign PIO[3] = pmod_dir[3] ? pmod_data[3] : 1'bz;
  assign PIO[4] = pmod_dir[4] ? pmod_data[4] : 1'bz;
  assign PIO[5] = pmod_dir[5] ? pmod_data[5] : 1'bz;
  assign PIO[6] = pmod_dir[6] ? pmod_data[6] : 1'bz;
  assign PIO[7] = pmod_dir[7] ? pmod_data[7] : 1'bz;

// =============================================================
// PLL: 48MHz
// =============================================================
  wire pclk;                      // System clock
  wire locked;                    // is used as /reset

  localparam MHZ = 48;
  pll myPLL (
    .inclk0(CLK12M),
    .c0(pclk),
    .locked(locked)
  );

// =============================================================
// Program-Memory: Dual ported 16K * 16 byte addressable
// =============================================================
  // Port A: Program area - only read ($0000-$7FFF - word size - addr[0]=0)
  wire [`MEMADDR-1:0] code_addr; // directly connected to PC
  wire [`WIDTH-1:0] insn;        // used for decoder

  // Port B: Data read/write ($0000-$7FFF - byte or word size - always readable)
  wire [`WIDTH-1:0] mem_din;     // data from memory
  wire [`WIDTH-1:0] mem_dout;    // data to memory
  wire [1:0]        mem_bsel;    // byte select for write will be latched
  wire              mem_wr;      // memory write

  // K1 - Memory and I/O read/write
  wire [`MEMADDR-1:0] k1_addr;   // address from K1 - will be latched to k1_addrL
  wire [`WIDTH-1:0]   k1_dout;   // data from K1
  reg  [`WIDTH-1:0]   k1_din;    // data to K1
  wire k1_bw;                    // Byte (0) or word (1) operation
  wire k1_wr;                    // memory or I/O write
  wire k1_rd;                    // memory or I/O read

  // Latch for read to keep it during next cycle - expect address in advance
  reg  [`MEMADDR-1:0] k1_addrL;  // used for read or write data
  assign mem_bsel[0] = k1_bw | (!k1_bw &  k1_addrL[0]); // word access or addr[0]=1)
  assign mem_bsel[1] = k1_bw | (!k1_bw & !k1_addrL[0]); // word access or addr[0]=0)
  assign mem_wr = k1_wr & !k1_addr[15];
  assign mem_dout = k1_bw ? k1_dout : (mem_bsel[1] ? {k1_dout[7:0], {8{1'b0}}} : {{8{1'b0}}, k1_dout[7:0]});

  prgmem myMem(
    .address_a(code_addr[`MEMADDR-2:1]),// Port A used for instruction read
    .address_b(k1_addr[`MEMADDR-2:1]),  // Port B for read/write data
    .byteena_a(2'b00),                  // no write to Port A
    .byteena_b(mem_bsel),               // Byte select only for write on Port B
    .clock(pclk),                     // New data at positive clock
    .data_a({`WIDTH{1'b0}}),            // not used
    .data_b(mem_dout),
    .wren_a(1'b0),                      // not used
    .wren_b(mem_wr),
    .q_a(insn),                         // Port A: instruction
    .q_b(mem_din)                       // Port B: data
  );

// =============================================================
// Internal Flash (32K)
// =============================================================
  reg  [ 2:0] ufm_ctrl     = 0; // UFM Bit 0: read; Bit 1: write; Bit 2: valid
  reg  [16:0] ufm_addr     = 0; // UFM address
  reg  [31:0] ufm_data     = 0; // UFM data for write
  wire [31:0] ufm_q;            // UFM data
  reg  [31:0] ufm_ql;           // UFM data latched for read
  wire        ufm_wait;         // UFM wait line
  wire        ufm_valid;        // UFM valid line
  reg  [ 2:0] ufm_csr_ctrl = 0; // CSR Bit 0: read; Bit 1: write; Bit 2: valid
  reg         ufm_csr_addr = 0; // CSR address
  reg  [31:0] ufm_csr_data = 0; // CSR data for write  
  wire [31:0] ufm_csr_q;        // CSR data for read
  reg  [31:0] ufm_csr_ql;       // CSR data latched for read
  reg         ufm_csr_read = 0; // Read CSR in the next cycle

  UFM ufm(
    .clock(pclk),
    .reset_n(locked),
    .avmm_data_addr(ufm_addr),
    .avmm_data_read(ufm_ctrl[0]),
    .avmm_data_readdata(ufm_q),
    .avmm_data_writedata(ufm_data),
    .avmm_data_write(ufm_ctrl[1]),
    .avmm_data_waitrequest(ufm_wait),
    .avmm_data_readdatavalid(ufm_valid),
    .avmm_data_burstcount(2'b01),
    .avmm_csr_addr(ufm_csr_addr),
    .avmm_csr_read(ufm_csr_ctrl[0]),
    .avmm_csr_writedata(ufm_csr_data),
    .avmm_csr_write(ufm_csr_ctrl[1]),
    .avmm_csr_readdata(ufm_csr_q)
  );	

// =============================================================
// UART (aktuell nur TX genutzt)
// =============================================================
  wire   RXD;                    // UART RX
  wire   TXD;                    // UART TX
  reg    RTS = 1'b1;             // Ready to Send Output
  reg    DSR = 1'b1;             // Data Set Ready Output

  assign RXD = BDBUS[0];         // Input
  assign BDBUS[1] = TXD;         // Output
  assign BDBUS[2] = RTS;         // Output
  assign BDBUS[5] = DSR;         // Output
  
  reg [15:0] uart_baudrate = MHZ * 1_000_000 / 115_200;

  wire rx_active;
  wire rx_error;
  wire rx_overflow;
  wire rx_full;
  reg  rx_read;
  wire rx_read_ff;
  wire [7:0] rx_data;

  wire tx_active;
  wire tx_error;
  wire tx_overflow;
  wire tx_full;
  reg  tx_write = 1'b0;
  wire tx_write_ff;
  reg [7:0] tx_data = 8'b0;

  myUART uart(
    .clk(pclk),                // The master clock
    .baudrate(uart_baudrate),  // Baudrate divider (speed = clk/baudrate)
    .reset(!locked),           // Synchronous reset
    .rx(RXD),                  // RX line
    .tx(TXD),                  // TX line

    .rx_active(rx_active),     // RX active.
    .rx_error(rx_error),       // RX receive error
    .rx_overflow(rx_overflow), // RX not read before next byte (overwritten)
    .rx_full(rx_full),         // A byte received
    .rx_read(rx_read),         // RX byte have read
    .rx_read_ff(rx_read_ff),   // wait for negative flag for next read
    .rx_data(rx_data),         // data from uart

    .tx_active(tx_active),     // TX active.
    .tx_error(tx_error),       // RX receive error
    .tx_overflow(tx_overflow), // RX not read before next byte (overwritten)
    .tx_full(tx_full),         // TX output buffer full
    .tx_write(tx_write),       // Assert to begin transmission
    .tx_write_ff(tx_write_ff), // TX acknowledge
    .tx_data(tx_data)          // data to uart
  );

// =============================================================
// Accelerometer
// =============================================================

  reg  [15:0] acc_baudrate = MHZ / 2; // Default: 1MHz
  wire [ 7:0] acc_in;          // Accelerometer interface (data and interrupts)
  reg  [ 7:0] acc_out = 8'b0;  // Data to Accelerometer SPI
  reg         acc_wr;          // Write to Accelerometer SPI
  wire        acc_busy;        // Accelerometer SPI is bussy
  reg         acc_cs = 1'b1;   // Accelerometer Chip Select
  
  assign SEN_CS = acc_cs;      // controlled by register
  
  mySPImaster acc(
    .reset(!locked),           // Synchronous reset
    .clk(pclk),                // The master clock
    .spi_baudrate(acc_baudrate), // Baudrate divider (speed = clk/baudrate/2)
    .spi_clk(SEN_SPC),         // Clock line
    .spi_busy(acc_busy),       // Busy
    .spi_mosi(SEN_SDI),        // MOSI line
    .spi_wr(acc_wr),           // write to SPI will start transfer
    .spi_out_data(acc_out),    // Byte to transmit
    .spi_miso(SEN_SDO),        // MISO line
    .spi_in_data(acc_in)       // Received data
);


// =============================================================
// FORTH-Processor K1
// Interfaces: Memory, I/O, Clock and Reset
// =============================================================

  K1 _K1(
    .clk(pclk),            // Clock: 100MHz
    .resetq(locked),       // Reset
    .code_addr(code_addr), // Program address
    .insn(insn),           // Next instruction
    .mem_addr(k1_addr),    // Memory address for read and write
    .mem_bw(k1_bw),        // Byte (0) or Word (1) access
    .dout(k1_dout),        // Data from K1
    .din(k1_din),          // Data to K1
    .mem_wr(k1_wr),        // Write to memory or IO
    .mem_rd(k1_rd)         // Read from memory or IO
  );

// =============================================================
// Assignments for the next posedge pclk
// =============================================================

  wire sel_mem = !k1_addrL[15]    & (k1_rd | k1_wr ); // $0000-$7FFF is selected
  wire sel_io  = &k1_addrL[15:12] & (k1_rd | k1_wr ); // $FF00-$FFFF is selected
  
  // k1_din
  always @* begin
    if(k1_rd & sel_mem) begin
      casez (mem_bsel[1:0])
        // Read from Memory (High-Endien: first high-byte then low-byte
        2'b01:   k1_din = {{(`WIDTH-8){1'b0}}, mem_din[7:0]};
        2'b10:   k1_din = {{(`WIDTH-8){1'b0}}, mem_din[15:8]};
        2'b11:   k1_din = {mem_din};
        default: k1_din = {16'h5aa5};
	    endcase
	 end else if(k1_rd & sel_io) begin
      casez (k1_addrL[5:1])
        // Read from UART ($Fxx0 = UART_Baudrate, $FF02 = Status, $FF04 = Data)
        5'b00000: k1_din = uart_baudrate;
        5'b00001: k1_din = {8'd0, BDBUS[5:0], rx_full, !tx_full};
	     5'b00010: k1_din = {8'd0, rx_data};
	     // Read from LED-Port ($Fxx6)
        5'b00011: k1_din = {7'd0, USER_BTN, ledreg};
        // Reading Counter ($Fxx8 = Low, $FFF0A = High)
        5'b00100: k1_din = counter[15:0];
        5'b00101: k1_din = counter[31:16];
        // PMOD Connector ($Fx0C = Data, $FF0E = Direction)
        5'b00110: k1_din = {{(`WIDTH-8){1'b0}}, PIO[7:0]};
        5'b00111: k1_din = {{(`WIDTH-8){1'b0}}, pmod_dir[7:0]};
	 	  // Accelerometer ($Fx10 = Baudrate, $FF12 = Data/Status)
        5'b01000: k1_din = acc_baudrate;
        5'b01001: k1_din = {acc_busy, {(`WIDTH-12){1'b0}}, SEN_INT2, SEN_INT1, acc_cs, acc_in};
        // Read from UFM $FF2x
        5'b10000: k1_din = {{(`WIDTH-12){1'b0}}, ufm_wait, ufm_ctrl, 7'b0, ufm_addr[16]};
        5'b10001: k1_din = ufm_addr[15:0];
        5'b10010: k1_din = ufm_ql[15: 0];
	     5'b10011: k1_din = ufm_ql[31:16];
        5'b10100: k1_din = {{(`WIDTH-13){1'b0}}, ufm_csr_ctrl, 8'b0};
        5'b10101: k1_din = {{(`WIDTH-1){1'b0}}, ufm_csr_addr};
        5'b10110: k1_din = ufm_csr_ql[15: 0];
	     5'b10111: k1_din = ufm_csr_ql[31:16];
        default:  k1_din = {16'h5aa5};
      endcase
	 end else begin
	               k1_din = {16'h5aa5};
	 end
  end

// =============================================================
// Main Loop
// =============================================================

  // Initialisation: LEDs off
  initial begin
    ledreg = 0;
    counter = 0;
  end

  
  always @(posedge pclk) begin // Using rising edge of clock
    // Counter use system clock
     counter <= counter + 1;

    // Latch adress and flags during read cycle
    k1_addrL  <= k1_addr;      // Adress for read

	 if(sel_io)begin
      casez ({k1_wr, k1_rd, k1_addrL[5:1]})
        // Read from UART
	     7'b01_00010: rx_read <= {1'b1};
        // Write to UART
        7'b10_00000: uart_baudrate <= k1_dout;
	     7'b10_00001: {DSR, RTS} <= {k1_dout[7], k1_dout[4]};
	     7'b10_00010: {tx_write, tx_data} <= {1'b1, k1_dout[7:0]};
        // Write to Ports
        7'b10_00011: ledreg    <= k1_dout[7:0]; // $FF06: LEDS
        7'b10_00110: pmod_data <= k1_dout[7:0]; // $FF0C: PMOD data
        7'b10_00111: pmod_dir  <= k1_dout[7:0]; // $FF0E: PMOD direction
		  // Accelerometer ($Fx10 = Baudrate, $Fx12 = Data/Status = New data only if acc_cs still 0)
        7'b10_01000: acc_baudrate <= k1_dout;
        7'b10_01001: begin
          if (!acc_cs & !k1_dout[8]) begin
            acc_out <= k1_dout[7:0];
            acc_wr  <= 1'b1;
	  	    end else begin
            acc_cs  <= k1_dout[8];
            acc_wr  <= 1'b0;
          end
        end
        // Read from UFM
        7'b01_10010: ufm_ctrl[2] <= 1'b0;     // $FF24: Read UFM data low reset valid flag
        7'b01_10110: ufm_csr_ctrl[2] <= 1'b0; // $FF2C: Read CSR data low reset valid flag
        // Write to UFM
        7'b10_10000: {ufm_ctrl[1:0], ufm_addr[16]} <= {k1_dout[9:8], k1_dout[0]}; // $FF20: UFM CTRL
        7'b10_10001: ufm_addr <= {ufm_addr[16], k1_dout[15:0]}; // $FF22: UFM address
        7'b10_10010: ufm_data[15: 0] <= k1_dout[15:0];      // $FF24: UFM data[15:0]
        7'b10_10011: ufm_data[31:16] <= k1_dout[15:0];      // $FF26: UFM data[31:15]
        7'b10_10100: ufm_csr_ctrl <= k1_dout[9:8];          // $FF28: CSR CTRL
        7'b10_10101: ufm_csr_addr <= k1_dout[0];            // $FF2A: CSR address
        7'b10_10110: ufm_csr_data[15: 0] <= k1_dout[15:0];  // $FF2C: CSR data[15:0]
        7'b10_10111: ufm_csr_data[31:16] <= k1_dout[15:0];  // $FF2E: CSR data[31:16]
        default:     {rx_read, tx_write, acc_wr} <= {1'b0, 1'b0, 1'b0};
      endcase
	 end else begin
      // Get data from UFM
      if(ufm_valid) begin // ufm_valid ?
        ufm_ql <= ufm_q;
        ufm_ctrl[2] <= 1;
		  ufm_ctrl[0] <= 0;
      end
		// Read from CSR ?
	   if(ufm_csr_ctrl[0]) begin
		  ufm_csr_read <= 1;
		  ufm_csr_ctrl[0] <= 0;
		end
      // Get data from UFM_CSR
	   if(ufm_csr_read) begin 
        ufm_csr_ql <= ufm_csr_q;
        ufm_csr_ctrl[2] <= 1;
		  ufm_csr_read <= 0;
      end
      // Write to UFM
      if(ufm_ctrl[1] & !ufm_wait) begin
		   ufm_ctrl[1] <= 0;
      end
		// Write to UFM_CSR
      if(ufm_csr_ctrl[1]) begin
		   ufm_csr_ctrl[1] <= 0;
      end
      {rx_read, tx_write, acc_wr} <= {1'b0, 1'b0, 1'b0};
	 end
	 
  end

endmodule // MAX1000_K1
